<?php

namespace App\Http\Controllers\PaymentGateway;

use Illuminate\Http\Request;
use CoinGate\Client;
use App\Http\Controllers\Controller;
use App\Facades\UtilityFacades;
use App\Models\Form;
use App\Models\FormValue;
use Hashids\Hashids;
use Illuminate\Support\Facades\Crypt;
use Exception;

class CoingateController extends Controller
{
    protected $client;

    public function __construct()
    {
        $this->client = new Client(
            UtilityFacades::getsettings('coingate_auth_token'),
            UtilityFacades::getsettings('coingate_environment')
        );
    }

    public function coingatePaymentPrepare(Request $request)
    {

        $allowedCurrencies = ['USD', 'EUR', 'GBP', 'BTC', 'USDT'];
        if (!in_array($request->cg_currency, $allowedCurrencies)) {
            return redirect()->back()->with('failed', __('Selected currency is not supported by CoinGate.'));
        }

        $params = [
            'order_id'         => 'CG-' . time() . '-' . rand(1000, 9999),
            'price_amount'     => $request->cg_amount,
            'price_currency'   => $request->cg_currency,
            'receive_currency' => $request->cg_currency,
            'callback_url'     => route('coingatecallback', Crypt::encrypt([
                'form_id' => $request->cg_form_id,
                'submit_type' => $request->cg_submit_type
            ])),
            'cancel_url'       => route('coingatecallback', Crypt::encrypt([
                'form_id' => $request->cg_form_id,
                'status' => 'failed',
                'submit_type' => $request->cg_submit_type
            ])),
            'success_url'      => route('coingatecallback', Crypt::encrypt([
                'form_id' => $request->cg_form_id,
                'status' => 'successfull',
                'submit_type' => $request->cg_submit_type
            ])),
        ];

        $order = $this->client->order->create($params);
        if (!$order) {
            return redirect()->back()->with('failed', __('Failed to create CoinGate order'));
        }
        $formvalue = FormValue::where('form_id', $request->cg_form_id)->latest('id')->first();
        $formvalue?->update(['transaction_id' => $order->id]);

        return redirect($order->payment_url);
    }

    public function coingatePlanGetPayment($data)
    {
        try {
            $data = Crypt::decrypt($data);
            if (!isset($data['form_id'])) {
                return redirect()->back()->with('failed', __('Invalid form data.'));
            }
            $form      = Form::findOrFail($data['form_id']);
            $formvalue = FormValue::where('form_id', $form->id)->latest('id')->first();

            if (!$formvalue) {
                return redirect()->back()->with('failed', __('Form submission not found.'));
            }
            $order = $this->client->order->get($formvalue->transaction_id ?? '');
            $formvalue->update([
                'transaction_id'  => $order->id ?? $formvalue->transaction_id,
                'currency_symbol' => $form->currency_symbol ?? '$',
                'currency_name'   => $order->price_currency ?? $form->currency_name ?? 'USD',
                'amount'          => $order->price_amount ?? $form->amount,
                'status'          => $order->status === 'paid' ? 'successfull' : 'failed',
                'payment_type'    => 'Coingate',
            ]);

            $id            = (new Hashids('', 20))->encodeHex($form->id);
            $redirectRoute = $data['submit_type'] === 'public_fill' ? redirect()->route('forms.survey', $id) : redirect()->back();
            if (isset($order->status) && $order->status !== 'paid') {
                return $redirectRoute->with('failed', __('CoinGate payment failed Status: ') . $order->status);
            }
            return $redirectRoute->with('success', strip_tags(trim($form->success_msg) ?: __('Form submitted successfully.')));
        } catch (Exception $e) {
            return redirect()->back()->with('failed', 'CoinGate payment processing failed: ' . $e->getMessage());
        }
    }
}
