<?php

namespace App\Http\Controllers;

use App\DataTables\FormStatusDataTable;
use App\Models\FormStatus;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class FormStatusController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(FormStatusDataTable $dataTable)
    {
        if (Auth::user()->can('manage-form-status')) {
            return $dataTable->render('form-status.index');
        } else {
            return redirect()->back()->with('failed', __('Permission denied.'));
        }
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        if (Auth::user()->can('create-form-status')) {
            $view =  view('form-status.create');
            return ['html' => $view->render()];
        } else {
            return redirect()->back()->with('failed', __('Permission denied.'));
        }
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        if (Auth::user()->can('create-form-status')) {
            request()->validate([
                'name'    => 'required|max:50',
                'status'  => 'required',
                'color'   => 'required',
            ]);

            FormStatus::create([
                'name'      => $request->name,
                'color'     => $request->color,
                'status'    => ($request->status == '1' ?? 1),
                'created_by' => Auth::user()->id
            ]);
            if (preg_match('/\/forms\/\d+\/edit$/', url()->previous())) {
                return redirect()->back()->with('success', __('Form Status Created Successfully!'));
            } else {
                return redirect()->route('form-status.index')->with('success', __('Form Status Created Successfully!'));
            }
        } else {
            return redirect()->back()->with('failed', __('Permission denied.'));
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(FormStatus $formStatus)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit($id)
    {
        if (Auth::user()->can('edit-form-status')) {
            $formStatus = FormStatus::find($id);
            $view = view('form-status.edit', compact('formStatus'));
            return ['html' => $view->render()];
        } else {
            return redirect()->back()->with('failed', __('Permission denied.'));
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        if (Auth::user()->can('edit-form-status')) {
            request()->validate([
                'name'   => 'required|max:50',
                'status' => 'required',
                'color'  => 'required'
            ]);

            $formStatus = FormStatus::find($id);
            $formStatus->update([
                'name'      => $request->name,
                'color'     => $request->color,
                'status'    => $request->status,
                'created_by'=> Auth::user()->id
            ]);
            return redirect()->route('form-status.index')->with('success', __('Form Status Updated Successfully!'));
        } else {
            return redirect()->back()->with('failed', __('Permission denied.'));
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        if (Auth::user()->can('delete-form-status')) {
            $formStatus = FormStatus::find($id);
            $formStatus->delete();
            return redirect()->route('form-status.index')->with('success', __('Form Status Deleted Successfully!'));
        } else {
            return redirect()->back()->with('failed', __('Permission denied.'));
        }
    }

    public function formStatusChange(Request $request, $id)
    {
        $formStatus = FormStatus::find($id);
        $input = ($request->value == "true") ? 1 : 0;
        if ($formStatus) {
            $formStatus->status = $input;
            $formStatus->save();
        }
        return response()->json(['is_success' => true, 'message' => __('Form status changed successfully.')]);
    }
}
