<?php

namespace App\Http\Controllers;

use App\DataTables\FormsDataTable;
use App\Facades\UtilityFacades;
use App\Mail\FormSubmitEmail;
use App\Mail\Thanksmail;
use App\Models\AssignFormsRoles;
use App\Models\AssignFormsUsers;
use App\Models\DashboardWidget;
use App\Models\Feedback;
use App\Models\Form;
use App\Models\FormCategory;
use App\Models\FormComments;
use App\Models\FormCommentsReply;
use App\Models\FormIntegrationSetting;
use App\Models\formRule;
use App\Models\FormStatus;
use App\Models\FormTemplate;
use App\Models\FormValue;
use App\Models\NotificationsSetting;
use App\Models\User;
use App\Models\UserForm;
use App\Notifications\CreateForm;
use App\Notifications\NewSurveyDetails;
use App\Rules\CommaSeparatedEmails;
use Carbon\Carbon;
use Exception;
use Google\Service\CloudAsset\Resource\Feeds;
use Hashids\Hashids;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Storage;
use Spatie\Permission\Models\Role;
use Stripe\Stripe as StripeStripe;
use Illuminate\Support\Facades\Mail;

class FormController extends Controller
{
    public function index(FormsDataTable $dataTable, Request $request)
    {
        if (Auth::user()->can('manage-form')) {
            if (Auth::user()->forms_grid_view == 1) {
                return redirect()->route('grid.form.view', 'view');
            }
            $categories = FormCategory::where('status', 1)->pluck('name', 'id');
            $roles      = Role::pluck('name', 'id');

            if (Auth::user()->type == 'Admin' || Auth::user()->can('access-all-form')) {
                $trashForm  = Form::onlyTrashed()->count();
                $form       = Form::count();
            } else {
                $role_id    = Auth::user()->roles->first()->id;
                $form       = Form::getAllAssignForm($role_id)->count();
                $trashForm  = Form::getAllAssignForm($role_id)->onlyTrashed()->count();
            }
            return $dataTable->render('form.index', compact('categories', 'roles', 'trashForm', 'form'));
        } else {
            return redirect()->back()->with('failed', __('Permission denied.'));
        }
    }

    public function addForm()
    {
        $formTemplates = FormTemplate::where('created_by', Auth::user()->id)->where('json', '!=', null)->where('status', 1)->get();
        return view('form.add', compact('formTemplates'));
    }

    public function create()
    {
        if (Auth::user()->can('create-form')) {
            $users    = User::where('id', '!=', 1)->pluck('name', 'id');
            $roles    = Role::where('name', '!=', 'Super Admin')->orwhere('name', Auth::user()->type)->pluck('name', 'id');
            $category = FormCategory::where('status', 1)->pluck('name', 'id');
            $status   = FormStatus::where('status', 1)->pluck('name', 'id');
            return view('form.create', compact('roles', 'users', 'category', 'status'));
        } else {
            return response()->json(['failed' => __('Permission denied.')], 401);
        }
    }

    public function useFormtemplate($id)
    {
        $formtemplate = FormTemplate::find($id);
        $form = Form::create([
            'title' => $formtemplate->title,
            'json'  => $formtemplate->json,
            'created_by' => Auth::user()->id,
        ]);
        return redirect()->route('forms.edit', $form->id)->with('success', __('Form created successfully.'));
    }

    public function store(Request $request)
    {
        if (Auth::user()->can('create-form')) {
            $request->validate([
                'title'       => 'required|string|max:191',
                'category_id' => 'required|integer|exists:form_categories,id',
                'form_logo'   => 'required|image|mimes:jpeg,jpg,png|max:2048',
                'email'       => ['nullable', new CommaSeparatedEmails],
                'bccemail'    => ['nullable', new CommaSeparatedEmails],
                'ccemail'     => ['nullable', new CommaSeparatedEmails],
                'form_status' => 'required|integer|exists:form_statuses,id',
                'assign_type' => 'required|in:role,user,public',
                'roles'       => 'required_if:assign_type,role|array|min:1',
                'users'       => 'required_if:assign_type,user|array|min:1',
            ], [
                'assign_type.in'    => 'The assign type must be one of: role, user.',
                'roles.required_if' => 'The role field is required when the assign type is role.',
                'roles.array'       => 'The roles field must be an array.',
                'roles.min'         => 'The roles field must have at least one role.',
                'users.required_if' => 'The user field is required when the assign type is user.',
                'users.array'       => 'The users field must be an array.',
                'users.min'         => 'The users field must have at least one user.',
            ]);
            $request->validate([
                'limit'             => 'required_if:limit_status,1',
                'form_password'     => 'required_if:password_enable,1',
                'set_end_date_time' => 'required_if:set_end_date,1',
            ], [
                'limit.required_if'         => 'The limit field is required when the Set Limit is on',
                'form_password.required_if' => 'The Password field is required when the Password Protection Enable is on',
                'set_end_date_time.required_if' => 'The Set End Date field is required when the Set End Date is on',
            ]);
            $formPassword = null;
            $password = 0;

            if (isset($request->password_enable) && $request->password_enable == 1) {
                $password       = 1;
                $formPassword   = Hash::make($request->form_password);
            }
            $fileName   = '';
            if (request()->hasFile('form_logo')) {
                $request->validate([
                    'form_logo'      => 'required|image|mimes:jpeg,jpg,png|max:2048'
                ]);
                $file     = $request->file('form_logo');
                $fileName = $file->store('form-logo');
            }

            $form               = new Form();
            $form->title        = $request->title;
            $form->logo         = $fileName;
            $form->description  = $request->form_description;
            $form->category_id  = $request->category_id;
            $form->email        = !empty($request->email) ? implode(',', $request->email) : null;
            $form->bccemail     = !empty($request->bccemail) ? implode(',', $request->bccemail) : null;
            $form->ccemail      = !empty($request->ccemail) ? implode(',', $request->ccemail) : null;
            $form->form_fill_edit_lock  = ($request->form_fill_edit_lock == 'on') ? '1' : '0';
            $form->enable_session_timer = ($request->enable_session_timer == 'on') ? '1' : '0';
            $form->allow_comments       = ($request->allow_comments == 'on') ? '1' : '0';
            $form->allow_share_section  = ($request->allow_share_section == 'on') ? '1' : '0';
            $form->feedback_enabled     = ($request->feedback_enabled == "on") ? '1' : "0";
            $form->json         = '';
            $form->success_msg  = $request->success_msg;
            $form->thanks_msg   = $request->thanks_msg;
            $form->set_end_date = ($request->set_end_date ?? 0) == 1 ? 1 : 0;
            $form->set_end_date_time = $request->set_end_date_time ? Carbon::parse($request->set_end_date_time)->toDateTimeString() : null;
            $form->created_by   = Auth::user()->id;
            $form->assign_type  = $request->assign_type;
            $form->limit_status = $request->limit_status;
            $form->limit        = $request->limit != '' ? $request->limit : null;
            $form->password_enabled = $password;
            $form->form_password    = $formPassword;
            $form->form_status      = $request->form_status;
            $form->save();

            if ($request->assign_type == 'role') {
                $form->assignRole($request->roles);
            }
            if ($request->assign_type == 'user') {
                $form->assignUser($request->users);
            }
            $form->assignFormRoles($request->roles);

            $notificationsSetting = NotificationsSetting::where('title', 'From Create')->first();
            if (isset($notificationsSetting) && $notificationsSetting->notify == '1') {
                $userSchema = User::where('type', '=', 'Admin')->first();
                $userSchema->notify(new CreateForm($form));
            }
            return redirect()->route('forms.index')->with('success', __('Form created successfully.'));
        } else {
            return redirect()->back()->with('failed', __('Permission denied.'));
        }
    }

    public function edit($id)
    {
        $usr      = Auth::user();
        $userRole = $usr->roles->first()->id;
        $formallowededit = UserForm::where('role_id', $userRole)->where('form_id', $id)->count();
        $form     = Form::find($id);
        if (!$form) {
            abort(404);
        }
        $formValueCount = FormValue::where('form_id', $form->id)->count();
        $hashids = new Hashids('', 20);
        $hashId  = $hashids->encodeHex($form->id);
        if (Auth::user()->can('edit-form') && $usr->type == 'Admin') {
            $next     = Form::where('id', '>', $form->id)->first();
            $previous = Form::where('id', '<', $form->id)->orderBy('id', 'desc')->first();
            $roles    = Role::where('name', '!=', 'Admin')->pluck('name', 'id');
            $formRole = $form->assignedroles->pluck('id')->toArray();
            $getFormRole = Role::pluck('name', 'id');
            $formUser    =  $form->assignedusers->pluck('id')->toArray();
            $GetformUser = User::where('id', '!=', 1)->pluck('name', 'id');
            $categories  = FormCategory::where('status', 1)->pluck('name', 'id');
            $status      = FormStatus::where('status', 1)->pluck('name', 'id');
            return view('form.edit', compact('form', 'roles', 'GetformUser', 'formUser', 'formRole', 'getFormRole', 'next', 'previous', 'categories', 'status', 'hashId', 'formValueCount'));
        } else {
            if (Auth::user()->can('access-all-form')) {
                $formCheck = Form::getAllForms()->pluck('id')->toArray();
            } else {
                $formCheck = Form::getAllAssignForm($userRole)->orWhere('created_by', Auth::user()->id)->pluck('id')->toArray();
            }

            if (Auth::user()->can('edit-form') && in_array($id, $formCheck) && $formallowededit > 0) {
                if (Auth::user()->can('access-all-form')) {
                    $next     =  Form::getAllForms()->where('id', '>', $form->id)->first();
                    $previous =  Form::getAllForms()->where('id', '>', $form->id)->first();
                } else {
                    $next     = Form::getAllAssignForm($userRole)->where('id', '>', $form->id)->first();
                    $previous = Form::getAllAssignForm($userRole)->where('id', '<', $form->id)->orderBy('id', 'desc')->first();
                }

                $roles    = Role::pluck('name', 'id');
                $formRole = $form->assignedroles->pluck('id')->toArray();
                $getFormRole  = Role::pluck('name', 'id');
                $formUser     = $form->assignedusers->pluck('id')->toArray();
                $GetformUser  = User::where('id', '!=', 1)->pluck('name', 'id');
                $categories   = FormCategory::where('status', 1)->pluck('name', 'id');
                $status       = FormStatus::where('status', 1)->pluck('name', 'id');

                return view('form.edit', compact('form', 'getFormRole', 'GetformUser', 'formUser', 'formRole', 'next', 'previous', 'categories', 'status', 'hashId', 'formValueCount'));
            } else {
                return redirect()->back()->with('failed', __('Permission denied.'));
            }
        }
    }

    public function update(Request $request, Form $form)
    {
        if (Auth::user()->can('edit-form')) {
            $request->validate([
                'title'       => 'required|string|max:191',
                'category_id' => 'required|integer|exists:form_categories,id',
                'email'       => ['nullable', new CommaSeparatedEmails],
                'form_status' => 'required|integer|exists:form_statuses,id',
                'bccemail'    => ['nullable', new CommaSeparatedEmails],
                'ccemail'     => ['nullable', new CommaSeparatedEmails],
                'assign_type' => 'required|in:role,user,public',
                'roles'       => 'required_if:assign_type,role|array|min:1',
                'users'       => 'required_if:assign_type,user|array|min:1',
            ], [
                'assign_type.in'    => 'The assign type must be one of: role, user.',
                'roles.required_if' => 'The role field is required when the assign type is role.',
                'roles.array'       => 'The roles field must be an array.',
                'roles.min'         => 'The roles field must have at least one role.',
                'users.required_if' => 'The user field is required when the assign type is user.',
                'users.array'       => 'The users field must be an array.',
                'users.min'         => 'The users field must have at least one user.',
            ]);
            $request->validate([
                'limit'             => 'required_if:limit_status,on',
                'form_password'     => 'required_if:password_enable,1',
                'set_end_date_time' => 'required_if:set_end_date,on',
            ], [
                'limit.required_if'         => 'The limit field is required when the Set Limit is on',
                'form_password.required_if' => 'The Password field is required when the Password Protection Enable is on',
                'set_end_date_time.required_if' => 'The Set End Date field is required when the Set End Date is on',
            ]);
            if (request()->hasFile('form_logo')) {
                $request->validate([
                    'form_logo' => 'required|image|mimes:jpeg,jpg,png|max:2048',
                ]);
                $file           = $request->file('form_logo');
                $fileName       = $file->store('form-logo');
                $form->logo     = $fileName;
            }

            $password = 0;
            $formPassword = null;
            if (isset($request->password_enable) && $request->password_enable == 1 && !empty($request->form_password)) {
                request()->validate([
                    'password_enable' => 'required',
                    'form_password' => 'required'
                ]);
                $password = 1;
                $formPassword = Hash::make($request->form_password);
            }

            $form->title       = $request->title;
            $form->success_msg = $request->success_msg;
            $form->thanks_msg  = $request->thanks_msg;
            $form->category_id = $request->category_id;
            $form->description = $request->form_description;
            $form->email       = !empty($request->email) ? implode(',', $request->email) : null;
            $form->bccemail    = !empty($request->bccemail) ? implode(',', $request->bccemail) : null;
            $form->ccemail     = !empty($request->ccemail) ? implode(',', $request->ccemail) : null;
            $form->form_fill_edit_lock  = ($request->form_fill_edit_lock == 'on') ? '1' : '0';
            $form->enable_session_timer = ($request->enable_session_timer == 'on') ? '1' : '0';
            $form->allow_comments       = ($request->allow_comments == 'on') ? '1' : '0';
            $form->allow_share_section  = ($request->allow_share_section == 'on') ? '1' : '0';
            $form->feedback_enabled  = ($request->feedback_enabled == "on") ? '1' : "0";
            $form->set_end_date      = ($request->set_end_date ?? 0) == 1 ? 1 : 0;
            $form->set_end_date_time = $request->set_end_date_time ? Carbon::parse($request->set_end_date_time)->toDateTimeString() : null;
            $form->created_by   = Auth::user()->id;
            $form->limit_status = $request->limit_status == 'on' ? '1' : '0';
            $form->limit        = $request->limit;
            $form->assign_type  = $request->assign_type;
            $form->password_enabled = $password;
            $form->form_password = $formPassword;
            $form->form_status   = $request->form_status;
            $form->save();

            if ($request->assign_type == 'role') {
                AssignFormsUsers::where('form_id', $form->id)->delete();
                $form->assignRole($request->roles);
            }
            if ($request->assign_type == 'user') {
                    AssignFormsRoles::where('form_id', $form->id)->delete();
                    $form->assignUser($request->users);
                }
            if ($request->assign_type == 'public') {
                AssignFormsRoles::where('form_id', $form->id)->delete();
                AssignFormsUsers::where('form_id', $form->id)->delete();
            }
            $form->assignFormRoles($request->roles);
            return redirect()->route('forms.index')->with('success', __('Form updated successfully.'));
        } else {
            return redirect()->back()->with('failed', __('Permission denied.'));
        }
    }

    public function destroy(Form $form)
    {
        if (Auth::user()->can('delete-form')) {
            if ($form) {
                Feedback::where('form_id', $form->id)->delete();
                Form::where('id', $form->id)->delete();
                FormValue::where('form_id', $form->id)->delete();
            }
            return redirect()->back()->with('success', __('Form deleted successfully'));
        } else {
            return redirect()->back()->with('failed', __('Permission denied.'));
        }
    }

    public function gridView($slug = '')
    {
        $authUser = Auth::user();
        $authUser->update(['forms_grid_view' => $slug ? 1 : 0]);
        if ($authUser->forms_grid_view == 0) {
            return redirect()->route('forms.index');
        }
        $role_id = $authUser->roles->first()->id;
        $user_id = $authUser->id;
        $forms = ($authUser->type == 'Admin' || $authUser->can('access-all-form')) ? Form::paginate(11)
            :  Form::where(function ($query) use ($role_id, $user_id) {
                $query->whereIn('forms.id', function ($query) use ($role_id) {
                    $query->select('form_id')
                        ->from('assign_forms_roles')
                        ->where('role_id', $role_id);
                })
                    ->orWhereIn('forms.id', function ($query) use ($user_id) {
                        $query->select('form_id')
                            ->from('assign_forms_users')
                            ->where('user_id', $user_id);
                    })
                    ->orWhere('forms.assign_type', 'public');
            })
            ->orderBy('forms.created_at', 'asc')->paginate(11);

        return view('form.grid-view', compact('forms'));
    }

    public function design($id)
    {
        if (!Auth::user()->can('design-form')) {
            return back()->with('failed', __('Permission denied.'));
        }
        $form = Form::find($id);
        return $form ? view('form.design', compact('form')) : redirect()->back()->with('failed', __('Form not found.'));
    }

    public function designUpdate(Request $request, $id)
    {
        if (!Auth::user()->can('design-form')) {
            return redirect()->back()->with('failed', __('Permission denied.'));
        }
        $form = Form::find($id);
        if ($redirect = $this->validateFormOrRedirect($form)) {
            return $redirect;
        }
        $form->json = $request->json;
        $fieldName  = json_decode($request->json);
        $arr        = [];
        foreach ($fieldName[0] as $k => $fields) {
            if ($fields->type == "header" || $fields->type == "paragraph") {
                $arr[$k] = $fields->type ?? '';
            } else {
                $arr[$k] = $fields->name ?? '';
            }
        }
        $value = DashboardWidget::where('form_id', $form->id)->pluck('field_name', 'id');
        foreach ($value  as $key => $v) {
            if (!in_array($v, $arr)) {
                DashboardWidget::find($key)?->delete();
            }
        }
        $form->save();
        return redirect()->route('forms.index')->with('success', __('Form updated successfully.'));
    }

    public function fill($id)
    {
        if (!Auth::user()->can('fill-form')) {
            return redirect()->back()->with('failed', __('Permission denied.'));
        }
        $userRole = Auth::user()->roles->first()->id;
        if (Auth::user()->type != 'Admin') {
            if (Auth::user()->can('access-all-form')) {
                $formCheck = Form::getAllForms()->pluck('id')->toArray();
            } else {
                $formCheck = Form::getAllAssignForm($userRole)->pluck('id')->toArray();
            }
            if (!in_array($id, $formCheck)) {
                abort(404);
            }
        }

        $form = Form::find($id);
        if ($redirect = $this->validateFormOrRedirect($form)) {
            return $redirect;
        }
        session(['formId' => $form->id]);
        $array = $form->getFormArray();
        return view('form.fill', compact('form', 'array'));
    }

    public function publicFill($id)
    {
        $hashids = new Hashids('', 20);
        $id      = $hashids->decodeHex($id);
        if ($id) {
            $form = Form::find($id);
            if ($redirect = $this->validateFormOrRedirect($form)) {
                return $redirect;
            }
            $todayDate  = Carbon::now()->toDateTimeString();
            if ($form->is_active == 1) {
                if ($form->set_end_date != '0') {
                    if ($form->set_end_date_time && $form->set_end_date_time < $todayDate) {
                        abort(404);
                    }
                }
                $array = $form->getFormArray();
                return view('form.public-fill', compact('form', 'array'));
            } else {
                abort(404);
            }
        } else {
            abort(404);
        }
    }

    public function qrCode($id)
    {
        $hashids = new Hashids('', 20);
        $id      = $hashids->decodeHex($id);
        $form    = Form::find($id);
        if ($redirect = $this->validateFormOrRedirect($form)) {
            return $redirect;
        }
        $view =  view('form.public-fill-qr', compact('form'));
        return ['html' => $view->render()];
    }

    public function fillStore(Request $request, $id)
    {
        $form = Form::find($id);
        if (!$form || empty($form)) {
            $message = __('Form not found');
            return isset($request->ajax)
                ? response()->json(['is_success' => false, 'message' => $message], 200)
                : redirect()->back()->with('failed', $message);
        }
        $formValueCount = FormValue::where('form_id', $form->id)->count();
        $ipData         = $request->input('ip_data');
        sleep(2);
        $ipDataArray = json_decode($ipData, true);
        $loc         = explode(',', $ipDataArray['loc']);

        if (UtilityFacades::getsettings('captcha_enable') == 'on') {
            if (UtilityFacades::getsettings('captcha') == 'hcaptcha') {
                if (empty($_POST['h-captcha-response'])) {
                    if (isset($request->ajax)) {
                        return response()->json(['is_success' => false, 'message' => __('Please check hcaptcha.')]);
                    } else {
                        return back()->with('failed', __('Please check hcaptcha.'));
                    }
                }
            }
            if (UtilityFacades::getsettings('captcha') == 'recaptcha') {
                if (empty($_POST['g-recaptcha-response'])) {
                    if (isset($request->ajax)) {
                        return response()->json(['is_success' => false, 'message' => __('Please check reCAPTCHA.')]);
                    } else {
                        return back()->with('failed', __('Please check reCAPTCHA.'));
                    }
                }
            }
        }
        $clientEmails = [];
        if ($request->form_value_id) {
            $formValue = FormValue::find($request->form_value_id);
            $array = json_decode($formValue->json);
        } else {
            $array = $form->getFormArray();
        }
        foreach ($array as  &$rows) {
            foreach ($rows as &$row) {
                if ($row->type == 'checkbox-group') {
                    foreach ($row->values as &$checkboxvalue) {
                        if (is_array($request->{$row->name}) && in_array($checkboxvalue->value, $request->{$row->name})) {
                            $checkboxvalue->selected = 1;
                        } else {
                            if (isset($checkboxvalue->selected)) {
                                unset($checkboxvalue->selected);
                            }
                        }
                    }
                } elseif ($row->type == 'file') {
                    $existingValues = $row->value ?? [];
                    if ($row->subtype == "fineuploader") {
                        $fileSize   = number_format($row->max_file_size_mb / 1073742848, 2);
                        $fileLimit  = $row->max_file_size_mb / 1024;
                        if ($fileSize < $fileLimit) {
                            $newValues = [];
                            if ($request->input($row->name)) {
                                $newFileValues   = explode(',', $request->input($row->name));
                                foreach ($newFileValues as $file) {
                                    $newValues[] = $file;
                                }
                            }
                            if (empty($newValues)) {
                                $row->value = $existingValues;
                            } else {
                                if ($row->multiple) {
                                    $row->value = array_merge($existingValues, $newValues);
                                } else {
                                    $row->value = $newValues;
                                }
                            }
                        } else {
                            return response()->json(['is_success' => false, 'message' => __("Please upload maximum $row->max_file_size_mb MB file size.")], 200);
                        }
                    } else {
                        if ($row->file_extention == 'pdf') {
                            $allowedFileExtension = ['pdf', 'pdfa', 'fdf', 'xdp', 'xfa', 'pdx', 'pdp', 'pdfxml', 'pdxox'];
                        } else if ($row->file_extention == 'image') {
                            $allowedFileExtension = ['jpeg', 'jpg', 'png'];
                        } else if ($row->file_extention == 'excel') {
                            $allowedFileExtension = ['xlsx', 'csv', 'xlsm', 'xltx', 'xlsb', 'xltm', 'xlw'];
                        }
                        $requiredExtension = implode(',', $allowedFileExtension);
                        $fileSizeLimit = $row->max_file_size_mb * 1024;

                        if ($request->hasFile($row->name)) {
                            if ($row->multiple) {
                                $uploadValues   = [];
                                $files          = $request->file($row->name);
                                foreach ($files as $file) {
                                    $extension  = $file->getClientOriginalExtension();
                                    $fileSize   = $file->getSize() / 1024;
                                    if (!in_array($extension, $allowedFileExtension) || $fileSize >= $fileSizeLimit) {
                                        $errorMessage = __("Invalid file type or size. Please upload files with extensions: $requiredExtension and maximum size $row->max_file_size_mb MB.");
                                        return $request->ajax ? response()->json(['is_success' => false, 'message' => $errorMessage], 200) : redirect()->back()->with('failed', $errorMessage);
                                    }
                                    if ($extension == 'csv') {
                                        $name           = \Str::random(40) . '.' . $extension;
                                        $file->move(Storage::path('form-values/' . $form->id), $name);
                                        $uploadValues[] = 'form-values/' . $form->id . '/' . $name;
                                    } else {
                                        $path       = Storage::path("form-values/$form->id");
                                        $fileName   = $file->store('form-values/' . $form->id);
                                        if (config('filesystems.default') == 'local') {
                                            if (!file_exists($path)) {
                                                mkdir($path, 0777, true);
                                                chmod($path, 0777);
                                            }
                                            if (!file_exists(Storage::path($fileName))) {
                                                mkdir(Storage::path($fileName), 0777, true);
                                                chmod(Storage::path($fileName), 0777);
                                            }
                                        }
                                        $uploadValues[] = $fileName;
                                    }
                                }
                                $row->value = array_merge($existingValues, $uploadValues);
                            } else {
                                $file       = $request->file($row->name);
                                $extension  = $file->getClientOriginalExtension();
                                $fileSize   = $file->getSize() / 1024;
                                if (!in_array($extension, $allowedFileExtension) || $fileSize >= $fileSizeLimit) {
                                    $errorMessage = __("Invalid file type or size. Please upload files with extensions: $requiredExtension and maximum size $row->max_file_size_mb MB.");
                                    return $request->ajax ? response()->json(['is_success' => false, 'message' => $errorMessage], 200) : redirect()->back()->with('failed', $errorMessage);
                                }
                                if ($extension) {
                                    $path      = Storage::path("form-values/$form->id");
                                    $fileName  = $file->store('form-values/' . $form->id);
                                    if (!file_exists($path)) {
                                        mkdir($path, 0777, true);
                                        chmod($path, 0777);
                                    }
                                    if (!file_exists(Storage::path($fileName))) {
                                        mkdir(Storage::path($fileName), 0777, true);
                                        chmod(Storage::path($fileName), 0777);
                                    }
                                    $values = $fileName;
                                }
                                $row->value = $values;
                            }
                        }
                    }
                } elseif ($row->type == 'radio-group') {
                    foreach ($row->values as &$radiovalue) {
                        if ($radiovalue->value == $request->{$row->name}) {
                            $radiovalue->selected = 1;
                        } else {
                            if (isset($radiovalue->selected)) {
                                unset($radiovalue->selected);
                            }
                        }
                    }
                } elseif ($row->type == 'autocomplete') {
                    if (isset($row->multiple)) {
                        foreach ($row->values as &$autocompletevalue) {
                            if (is_array($request->{$row->name}) && in_array($autocompletevalue->value, $request->{$row->name})) {
                                $autocompletevalue->selected = 1;
                            } else {
                                if (isset($autocompletevalue->selected)) {
                                    unset($autocompletevalue->selected);
                                }
                            }
                        }
                    } else {
                        foreach ($row->values as &$autocompletevalue) {
                            if ($autocompletevalue->value == $request->autocomplete) {
                                $autocompletevalue->selected = 1;
                            } else {
                                if (isset($autocompletevalue->selected)) {
                                    unset($autocompletevalue->selected);
                                }
                                $row->value = $request->autocomplete;
                            }
                        }
                    }
                } elseif ($row->type == 'select') {
                    if ($row->multiple) {
                        foreach ($row->values as &$selectvalue) {
                            if (is_array($request->{$row->name}) && in_array($selectvalue->value, $request->{$row->name})) {
                                $selectvalue->selected = 1;
                            } else {
                                if (isset($selectvalue->selected)) {
                                    unset($selectvalue->selected);
                                }
                            }
                        }
                    } else {
                        foreach ($row->values as &$selectvalue) {
                            if ($selectvalue->value == $request->{$row->name}) {
                                $selectvalue->selected = 1;
                            } else {
                                if (isset($selectvalue->selected)) {
                                    unset($selectvalue->selected);
                                }
                            }
                        }
                    }
                } elseif ($row->type == 'date' || $row->type == 'number' || $row->type == 'textarea') {
                    $row->value = $request->{$row->name};
                } elseif ($row->type == 'text') {
                    $clientEmail = '';
                    if ($row->subtype == 'email') {
                        if (isset($row->is_client_email) && $row->is_client_email) {
                            $emailValue = $request->{$row->name};
                            if (isset($emailValue) && !empty($emailValue) && !filter_var($emailValue, FILTER_VALIDATE_EMAIL)) {
                                if (isset($request->ajax)) {
                                    return response()->json(['is_success' => false, 'message' => __('Please enter a valid email address.')], 200);
                                } else {
                                    return redirect()->back()->with('failed', __('Please enter a valid email address.'));
                                }
                            } else {
                                $clientEmails[] = $emailValue;
                            }
                        }
                    }
                    $row->value = $request->{$row->name};
                } elseif ($row->type == 'starRating') {
                    $row->value = $request->{$row->name};
                } elseif ($row->type == 'SignaturePad') {
                    if (property_exists($row, 'value')) {
                        $filepath = $row->value;
                        if ($request->{$row->name} == null) {
                            $url = $row->value;
                        } else {
                            if (file_exists(Storage::path($request->{$row->name}))) {
                                $url      = $request->{$row->name};
                                $path     = $url;
                                $imgUrl   = Storage::path($url);
                                $filePath = $imgUrl;
                            } else {
                                $imgUrl   = $request->{$row->name};
                                $path     = "form-values/$form->id/" . rand(1, 1000) . '.png';
                                $filePath = Storage::path($path);
                            }
                            $imageContent = file_get_contents($imgUrl);
                            $file = file_put_contents($filePath, $imageContent);
                        }
                        $row->value = $path;
                    } else {
                        if ($request->{$row->name} != null) {
                            if (!file_exists(Storage::path("form-values/$form->id"))) {
                                mkdir(Storage::path("form-values/$form->id"), 0777, true);
                                chmod(Storage::path("form-values/$form->id"), 0777);
                            }
                            $filepath     = "form-values/$form->id/" . rand(1, 1000) . '.png';
                            $url          = $request->{$row->name};
                            $imageContent = file_get_contents($url);
                            $filePath     = Storage::path($filepath);
                            $file         = file_put_contents($filePath, $imageContent);
                            $row->value   = $filepath;
                        }
                    }
                } elseif ($row->type == 'location') {
                    $row->value = $request->location;
                } elseif ($row->type == 'video') {
                    $row->value = $request->{$row->name};
                } elseif ($row->type == 'selfie') {
                    $file       = '';
                    $img        = $request->{$row->name};
                    if ($img) {
                        $folderPath = "form-selfie/";
                        $imageParts = explode(";base64,", $img);
                        if ($imageParts[0]) {
                            $imageBase64    = base64_decode($imageParts[1]);
                            $fileName       = uniqid() . '.png';
                            $file           = $folderPath . $fileName;
                            Storage::put($file, $imageBase64);
                        }
                        $row->value = $file;
                    }
                }
            }
        }

        if ($request->form_value_id) {
            $formValue->json = json_encode($array);
            $formValue->submited_forms_ip         = $ipDataArray['ip'];
            $formValue->submited_forms_country    = $ipDataArray['country'];
            $formValue->submited_forms_region     = $ipDataArray['region'];
            $formValue->submited_forms_city       = $ipDataArray['city'];
            $formValue->submited_forms_latitude   = $loc[0];
            $formValue->submited_forms_longitude  = $loc[1];
            $formValue->save();
        } else {
            if (Auth::user()) {
                $userId = Auth::user()->id;
            } else {
                $userId = NULL;
            }
            $data = [];
            if ($form->payment_status == 1) {
                if ($form->payment_type == 'stripe') {
                    StripeStripe::setApiKey(UtilityFacades::getsettings('STRIPE_SECRET', $form->created_by));
                    try {
                        $paymentIntent = \Stripe\PaymentIntent::create([
                            'amount'         => $form->amount * 100,
                            'currency'       => $form->currency_name,
                            'payment_method' => $request->payment_method_id,
                            'description'    => "Payment from " . config('app.name'),
                            'confirm'        => true,
                            'metadata' => [
                                'order_id' => $form->id,
                            ],
                            'automatic_payment_methods' => [
                                'enabled'         => true,
                                'allow_redirects' => 'never'
                            ],
                        ]);
                    } catch (Exception $e) {
                        return response()->json(['status' => false, 'message' => $e->getMessage()], 200);
                    }
                    if ($paymentIntent) {
                        $data['transaction_id']  = $paymentIntent->id;
                        $data['currency_symbol'] = $form->currency_symbol;
                        $data['currency_name']   = $form->currency_name;
                        $data['amount']          = $form->amount;
                        $data['status']          = ($paymentIntent->status === 'succeeded') ? 'successfull' : 'failed';
                        $data['payment_type']    = 'Stripe';
                    }
                } else if ($form->payment_type == 'razorpay') {
                    $response = Http::withBasicAuth(
                        UtilityFacades::getsettings('razorpay_key'),
                        UtilityFacades::getsettings('razorpay_secret')
                    )->get("https://api.razorpay.com/v1/payments/{$request->payment_id}");
                    $data['transaction_id']  = $request->payment_id;
                    $data['currency_symbol'] = $form->currency_symbol;
                    $data['currency_name']   = $form->currency_name;
                    $data['amount']          = $form->amount;
                    $data['status']          = $response->successful() ? 'successfull' : 'failed';
                    $data['payment_type']    = 'Razorpay';
                } else if ($form->payment_type == 'paypal') {
                    $data['transaction_id']  = $request->payment_id;
                    $data['currency_symbol'] = $form->currency_symbol;
                    $data['currency_name']   = $form->currency_name;
                    $data['amount']          = $form->amount;
                    $data['status']          = 'successfull';
                    $data['payment_type']    = 'Paypal';
                } else if ($form->payment_type == 'flutterwave') {
                    $data['transaction_id']  = $request->payment_id;
                    $data['currency_symbol'] = $form->currency_symbol;
                    $data['currency_name']   = $form->currency_name;
                    $data['amount']          = $form->amount;
                    $data['status']          = 'successfull';
                    $data['payment_type'] = 'Flutterwave';
                } else if ($form->payment_type == 'paytm') {
                    $data['transaction_id']  = $request->payment_id;
                    $data['currency_symbol'] = $form->currency_symbol;
                    $data['currency_name']   = $form->currency_name;
                    $data['amount']          = $form->amount;
                    $data['status']          = 'pending';
                    $data['payment_type']    = 'Paytm';
                } else if ($form->payment_type == 'paystack') {
                    $data['transaction_id']   = $request->payment_id;
                    $data['currency_symbol']  = $form->currency_symbol;
                    $data['currency_name']    = $form->currency_name;
                    $data['amount']           = $form->amount;
                    $data['status']           = 'successfull';
                    $data['payment_type'] = 'Paystack';
                } else if ($form->payment_type == 'payumoney') {
                    $data['transaction_id']   = $request->payment_id;
                    $data['currency_symbol']  = $form->currency_symbol;
                    $data['currency_name']    = $form->currency_name;
                    $data['amount']           = $form->amount;
                    $data['status']           = 'pending';
                    $data['payment_type'] = 'PayuMoney';
                } else if ($form->payment_type == 'mollie') {
                    $data['transaction_id']   = $request->payment_id;
                    $data['currency_symbol']  = $form->currency_symbol;
                    $data['currency_name']    = $form->currency_name;
                    $data['amount']           = $form->amount;
                    $data['status']           = 'pending';
                    $data['payment_type'] = 'Mollie';
                } else if ($form->payment_type == 'coingate') {
                    $data['status'] = 'pending';
                } else if ($form->payment_type == 'mercado') {
                    $data['status'] = 'pending';
                } elseif ($form->payment_type == 'offlinepayment') {
                    $data['currency_symbol']  = $form->currency_symbol;
                    $data['currency_name']    = $form->currency_name;
                    $data['amount']           = $form->amount;
                    $data['payment_type']     = 'offlinepayment';
                    $request->validate(['transfer_slip' => 'required']);
                    if ($request->file('transfer_slip')) {
                        $file = $request->file('transfer_slip');
                        $filename = $file->store('transfer-slip');
                        $data['transfer_slip'] = $filename;
                    }
                    if ($request->transfer_slip) {
                        $data['status']       = 'successfull';
                    } else {
                        $data['status']       = 'pending';
                    }
                }
            } else {
                $data['status'] = 'free';
            }

            $data['form_id']                   = $form->id;
            $data['user_id']                   = $userId;
            $data['json']                      = json_encode($array);
            $data['form_edit_lock_status']     = $form->form_fill_edit_lock;
            $data['form_status']               = $form->form_status;
            $data['submited_forms_ip']         = $ipDataArray['ip'];
            $data['submited_forms_country']    = $ipDataArray['country'];
            $data['submited_forms_region']     = $ipDataArray['region'];
            $data['submited_forms_city']       = $ipDataArray['city'];
            $data['submited_forms_latitude']   = $loc[0];
            $data['submited_forms_longitude']  = $loc[1];

            if ($form->limit_status == 0 || empty($form->limit)) {
                $formValue = FormValue::create($data);
            } elseif ($form->limit_status == 1 && !empty($form->limit) && $form->limit  > $formValueCount || $request->form_value_id != null) {
                $formValue = FormValue::create($data);
            } else {
                return response()->json(['is_success' => false, 'message' => __('Form submission limit is over')], 200);
            }
        }
        $formValueArray = json_decode($formValue->json);
        $user           = User::where('type', 'Admin')->first();
        $notificationsSetting = NotificationsSetting::where('title', 'new survey details')->first();
        if (isset($notificationsSetting)) {
            if ($notificationsSetting->notify == '1') {
                $user->notify(new NewSurveyDetails($form));
            }
            if (UtilityFacades::getsettings('email_setting_enable') == 'on') {
                try {
                    $emails = array_merge(
                        [$form->email],
                        array_filter(array_map('trim', explode(',', $form->ccemail ?? ''))),
                        array_filter(array_map('trim', explode(',', $form->bccemail ?? '')))
                    );

                    if ($notificationsSetting->email_notification == '1' && !empty($formValue)) {
                        foreach ($emails as $email) {
                            Mail::to($email)->send(new FormSubmitEmail($formValue, $formValueArray));
                        }
                    }
                    foreach ($clientEmails as $clientEmail) {
                        Mail::to($clientEmail)->send(new Thanksmail($formValue));
                    }
                } catch (\Exception $e) {
                    return isset($request->ajax) ? response()->json(['is_success' => false, 'message' => $e->getMessage()], 200) : redirect()->back()->with('failed', $e->getMessage());
                }
            }
        }
        Form::integrationFormData($form, $formValue);
        $successMsg = strip_tags($form->success_msg);

        if (isset($request->ajax)) {
            return response()->json([
                'is_success' => true,
                'message' => $successMsg,
                'formValueId' => $formValue->id,
                'redirect' => route('edit.form.values', $formValue->id)
            ], 200);
        }

        return redirect()->back()->with('success', $successMsg);
    }

    public function upload(Request $request)
    {
        if ($request->hasFile('upload')) {
            $fileName = $request->upload->store('editor');
            $CKEditorFuncNum = $request->input('CKEditorFuncNum');
            $url      = Storage::url($fileName);
            $msg      = __('Image uploaded successfully');
            $response = "<script>window.parent.CKEDITOR.tools.callFunction($CKEditorFuncNum, '$url', '$msg')</script>";
            @header('Content-type: text/html; charset=utf-8');
            echo $response;
        }
    }

    public function duplicate(Request $request)
    {
        if (Auth::user()->can('duplicate-form')) {
            $request->validate([
                'form_id' => 'required|exists:forms,id',
            ]);
            try {
                $originalForm   = Form::findOrFail($request->form_id);
                $newForm        = $originalForm->replicate();
                $newForm->title = $originalForm->title . ' (copy)';
                $newForm->created_by = Auth::id();
                $newForm->save();
                return redirect()->back()->with('success', __('Form duplicated successfully.'));
            } catch (\Exception $e) {
                return redirect()->back()->with('failed', __('Failed to duplicate form. Please try again.'));
            }
        } else {
            return redirect()->back()->with('failed', __('Permission denied.'));
        }
    }

    public function ckupload(Request $request)
    {
        if ($request->hasFile('upload')) {
            $originName = $request->file('upload')->getClientOriginalName();
            $fileName   = pathinfo($originName, PATHINFO_FILENAME);
            $extension  = $request->file('upload')->getClientOriginalExtension();
            $fileName   = $fileName . '_' . time() . '.' . $extension;
            $request->file('upload')->move(public_path('images'), $fileName);
            $CKEditorFuncNum = $request->input('CKEditorFuncNum');
            $url      = asset('images/' . $fileName);
            $msg      = __('Image uploaded successfully');
            $response = "<script>window.parent.CKEDITOR.tools.callFunction($CKEditorFuncNum, '$url', '$msg')</script>";
            @header('Content-type: text/html; charset=utf-8');
            echo $response;
        }
    }

    public function dropzone(Request $request, $id)
    {
        $allowedfileExtension   = [];
        if ($request->file_extention == 'pdf') {
            $allowedfileExtension = ['pdf', 'pdfa', 'fdf', 'xdp', 'xfa', 'pdx', 'pdp', 'pdfxml', 'pdxox'];
        } else if ($request->file_extention == 'image') {
            $allowedfileExtension = ['jpeg', 'jpg', 'png'];
        } else if ($request->file_extention == 'excel') {
            $allowedfileExtension = ['xlsx', 'csv', 'xlsm', 'xltx', 'xlsb', 'xltm', 'xlw'];
        } else {
            return response()->json(['errors' => 'Invalid file type selected.']);
        }

        if ($request->hasFile('file')) {
            $files = $request->file('file');
            $storedFiles = [];
            if (!is_array($files)) {
                $files = [$files];
            }

            foreach ($files as $file) {
                $extension = $file->getClientOriginalExtension();
                if (!$extension) {
                    $extension = $file->guessExtension();
                }

                if (in_array(strtolower($extension), $allowedfileExtension)) {
                    $filename      = $file->store('form-values/' . $id);
                    $storedFiles[] = $filename;
                } else {
                    return response()->json(['errors' => 'Only ' . implode(',', $allowedfileExtension) . ' file types are allowed.']);
                }
            }

            return response()->json(['success' => __('Files uploaded successfully.'), 'filename' => $storedFiles]);
        } else {
            return response()->json(['errors' => __('File not found.')]);
        }
    }

    public function formStatus(Request $request, $id)
    {
        $form  = Form::find($id);
        $input = ($request->value == "true") ? 1 : 0;
        if ($form) {
            $form->is_active = $input;
            $form->save();
        }
        return response()->json(['is_success' => true, 'message' => __('Form status changed successfully.')]);
    }

    public function formIntegration($id)
    {
        $form = Form::find($id);
        if ($redirect = $this->validateFormOrRedirect($form)) {
            return $redirect;
        }
        $formJsons       = json_decode($form->json);
        $slackSettings   = FormIntegrationSetting::where('key', 'slack_integration')->where('form_id', $form->id)->first();
        $slackJsons      = [];
        $slackFieldJsons = [];
        if ($slackSettings) {
            $slackFieldJsons = json_decode($slackSettings->field_json, true);
            $slackJsons      = json_decode($slackSettings->json, true);
        }
        $telegramSettings   = FormIntegrationSetting::where('key', 'telegram_integration')->where('form_id', $form->id)->first();
        $telegramJsons      = [];
        $telegramFieldJsons = [];
        if ($telegramSettings) {
            $telegramFieldJsons = json_decode($telegramSettings->field_json, true);
            $telegramJsons      = json_decode($telegramSettings->json, true);
        }
        $mailgunSettings    = FormIntegrationSetting::where('key', 'mailgun_integration')->where('form_id', $form->id)->first();
        $mailgunJsons       = [];
        $mailgunFieldJsons  = [];
        if ($mailgunSettings) {
            $mailgunFieldJsons = json_decode($mailgunSettings->field_json, true);
            $mailgunJsons      = json_decode($mailgunSettings->json, true);
        }
        $bulkgateSettings   = FormIntegrationSetting::where('key', 'bulkgate_integration')->where('form_id', $form->id)->first();
        $bulkgateJsons      = [];
        $bulkgateFieldJsons = [];
        if ($bulkgateSettings) {
            $bulkgateFieldJsons = json_decode($bulkgateSettings->field_json, true);
            $bulkgateJsons      = json_decode($bulkgateSettings->json, true);
        }
        $nexmoSettings   = FormIntegrationSetting::where('key', 'nexmo_integration')->where('form_id', $form->id)->first();
        $nexmoJsons      = [];
        $nexmoFieldJsons = [];
        if ($nexmoSettings) {
            $nexmoFieldJsons = json_decode($nexmoSettings->field_json, true);
            $nexmoJsons      = json_decode($nexmoSettings->json, true);
        }
        $fast2smsSettings   = FormIntegrationSetting::where('key', 'fast2sms_integration')->where('form_id', $form->id)->first();
        $fast2smsJsons      = [];
        $fast2smsFieldJsons = [];
        if ($fast2smsSettings) {
            $fast2smsFieldJsons = json_decode($fast2smsSettings->field_json, true);
            $fast2smsJsons      = json_decode($fast2smsSettings->json, true);
        }
        $vonageSettings   = FormIntegrationSetting::where('key', 'vonage_integration')->where('form_id', $form->id)->first();
        $vonageJsons      = [];
        $vonageFieldJsons = [];
        if ($vonageSettings) {
            $vonageFieldJsons = json_decode($vonageSettings->field_json, true);
            $vonageJsons      = json_decode($vonageSettings->json, true);
        }
        $sendgridSettings   = FormIntegrationSetting::where('key', 'sendgrid_integration')->where('form_id', $form->id)->first();
        $sendgridJsons      = [];
        $sendgridFieldJsons = [];
        if ($sendgridSettings) {
            $sendgridFieldJsons = json_decode($sendgridSettings->field_json, true);
            $sendgridJsons      = json_decode($sendgridSettings->json, true);
        }
        $twilioSettings   = FormIntegrationSetting::where('key', 'twilio_integration')->where('form_id', $form->id)->first();
        $twilioJsons      = [];
        $twilioFieldJsons = [];
        if ($twilioSettings) {
            $twilioFieldJsons = json_decode($twilioSettings->field_json, true);
            $twilioJsons      = json_decode($twilioSettings->json, true);
        }
        $textlocalSettings   = FormIntegrationSetting::where('key', 'textlocal_integration')->where('form_id', $form->id)->first();
        $textlocalJsons      = [];
        $textlocalFieldJsons = [];
        if ($textlocalSettings) {
            $textlocalFieldJsons = json_decode($textlocalSettings->field_json, true);
            $textlocalJsons      = json_decode($textlocalSettings->json, true);
        }
        $messenteSettings   = FormIntegrationSetting::where('key', 'messente_integration')->where('form_id', $form->id)->first();
        $messenteJsons      = [];
        $messenteFieldJsons = [];
        if ($messenteSettings) {
            $messenteFieldJsons = json_decode($messenteSettings->field_json, true);
            $messenteJsons      = json_decode($messenteSettings->json, true);
        }
        $smsgatewaySettings = FormIntegrationSetting::where('key', 'smsgateway_integration')->where('form_id', $form->id)->first();
        $smsgatewayJsons      = [];
        $smsgatewayFieldJsons = [];
        if ($smsgatewaySettings) {
            $smsgatewayFieldJsons = json_decode($smsgatewaySettings->field_json, true);
            $smsgatewayJsons = json_decode($smsgatewaySettings->json, true);
        }
        $clicktellSettings = FormIntegrationSetting::where('key', 'clicktell_integration')->where('form_id', $form->id)->first();
        $clicktellJsons      = [];
        $clicktellFieldJsons = [];
        if ($clicktellSettings) {
            $clicktellFieldJsons = json_decode($clicktellSettings->field_json, true);
            $clicktellJsons = json_decode($clicktellSettings->json, true);
        }
        $clockworkSettings = FormIntegrationSetting::where('key', 'clockwork_integration')->where('form_id', $form->id)->first();
        $clockworkJsons      = [];
        $clockworkFieldJsons = [];
        if ($clockworkSettings) {
            $clockworkFieldJsons = json_decode($clockworkSettings->field_json, true);
            $clockworkJsons = json_decode($clockworkSettings->json, true);
        }

        $salesforceSettings = FormIntegrationSetting::where('key', 'salesforce_integration')->where('form_id', $form->id)->first();
        $salesforceJsons      = [];
        $salesforceFieldJsons = [];
        if ($salesforceSettings) {
            $salesforceFieldJsons = json_decode($salesforceSettings->field_json, true);
            $salesforceJsons    = json_decode($salesforceSettings->json, true);
        }

        $hubspotSettings = FormIntegrationSetting::where('key', 'hubspot_integration')->where('form_id', $form->id)->first();
        $hubspotJsons      = [];
        $hubspotFieldJsons = [];
        if ($hubspotSettings) {
            $hubspotFieldJsons = json_decode($hubspotSettings->field_json, true);
            $hubspotJsons = json_decode($hubspotSettings->json, true);
        }

        $pipedriveSettings = FormIntegrationSetting::where('key', 'pipedrive_integration')->where('form_id', $form->id)->first();
        $pipedriveJsons      = [];
        $pipedriveFieldJsons = [];
        if ($pipedriveSettings) {
            $pipedriveFieldJsons = json_decode($pipedriveSettings->field_json, true);
            $pipedriveJsons = json_decode($pipedriveSettings->json, true);
        }

        return view('form.integration.index', compact('form', 'slackJsons', 'telegramJsons', 'mailgunJsons', 'bulkgateJsons', 'nexmoJsons', 'fast2smsJsons', 'vonageJsons', 'sendgridJsons', 'twilioJsons', 'textlocalJsons', 'messenteJsons', 'smsgatewayJsons', 'clicktellJsons', 'clockworkJsons', 'formJsons', 'salesforceJsons', 'hubspotJsons', 'pipedriveJsons', 'pipedriveFieldJsons', 'hubspotFieldJsons', 'salesforceFieldJsons', 'slackFieldJsons', 'telegramFieldJsons', 'mailgunFieldJsons', 'bulkgateFieldJsons', 'nexmoFieldJsons', 'fast2smsFieldJsons', 'vonageFieldJsons', 'sendgridFieldJsons', 'twilioFieldJsons', 'textlocalFieldJsons', 'messenteFieldJsons', 'smsgatewayFieldJsons', 'clicktellFieldJsons', 'clockworkFieldJsons'));
    }

    public function slackIntegration($id)
    {
        $form = Form::find($id);
        if ($redirect = $this->validateFormOrRedirect($form)) {
            return $redirect;
        }
        $formJsons = json_decode($form->json);
        return view('form.integration.slack', compact('form', 'formJsons'));
    }

    public function telegramIntegration($id)
    {
        $form = Form::find($id);
        if ($redirect = $this->validateFormOrRedirect($form)) {
            return $redirect;
        }
        $formJsons = json_decode($form->json);
        return view('form.integration.telegram', compact('form', 'formJsons'));
    }

    public function mailgunIntegration($id)
    {
        $form = Form::find($id);
        if ($redirect = $this->validateFormOrRedirect($form)) {
            return $redirect;
        }
        $formJsons = json_decode($form->json);
        return view('form.integration.mailgun', compact('form', 'formJsons'));
    }

    public function bulkgateIntegration($id)
    {
        $form = Form::find($id);
        if ($redirect = $this->validateFormOrRedirect($form)) {
            return $redirect;
        }
        $formJsons = json_decode($form->json);
        return view('form.integration.bulkgate', compact('form', 'formJsons'));
    }

    public function nexmoIntegration($id)
    {
        $form = Form::find($id);
        if ($redirect = $this->validateFormOrRedirect($form)) {
            return $redirect;
        }
        $formJsons = json_decode($form->json);
        return view('form.integration.nexmo', compact('form', 'formJsons'));
    }

    public function fast2smsIntegration($id)
    {
        $form = Form::find($id);
        if ($redirect = $this->validateFormOrRedirect($form)) {
            return $redirect;
        }
        $formJsons = json_decode($form->json);
        return view('form.integration.fast2sms', compact('form', 'formJsons'));
    }

    public function vonageIntegration($id)
    {
        $form = Form::find($id);
        if ($redirect = $this->validateFormOrRedirect($form)) {
            return $redirect;
        }
        $formJsons = json_decode($form->json);
        return view('form.integration.vonage', compact('form', 'formJsons'));
    }

    public function sendgridIntegration($id)
    {
        $form = Form::find($id);
        if ($redirect = $this->validateFormOrRedirect($form)) {
            return $redirect;
        }
        $formJsons = json_decode($form->json);
        return view('form.integration.sendgrid', compact('form', 'formJsons'));
    }

    public function twilioIntegration($id)
    {
        $form = Form::find($id);
        if ($redirect = $this->validateFormOrRedirect($form)) {
            return $redirect;
        }
        $formJsons = json_decode($form->json);
        return view('form.integration.twilio', compact('form', 'formJsons'));
    }

    public function textlocalIntegration($id)
    {
        $form = Form::find($id);
        if ($redirect = $this->validateFormOrRedirect($form)) {
            return $redirect;
        }
        $formJsons = json_decode($form->json);
        return view('form.integration.textlocal', compact('form', 'formJsons'));
    }

    public function messenteIntegration($id)
    {
        $form = Form::find($id);
        if ($redirect = $this->validateFormOrRedirect($form)) {
            return $redirect;
        }
        $formJsons = json_decode($form->json);
        return view('form.integration.messente', compact('form', 'formJsons'));
    }

    public function smsgatewayIntegration($id)
    {
        $form = Form::find($id);
        if ($redirect = $this->validateFormOrRedirect($form)) {
            return $redirect;
        }
        $formJsons = json_decode($form->json);
        return view('form.integration.smsgateway', compact('form', 'formJsons'));
    }

    public function clicktellIntegration($id)
    {
        $form = Form::find($id);
        if ($redirect = $this->validateFormOrRedirect($form)) {
            return $redirect;
        }
        $formJsons = json_decode($form->json);
        return view('form.integration.clicktell', compact('form', 'formJsons'));
    }

    public function clockworkIntegration($id)
    {
        $form = Form::find($id);
        if ($redirect = $this->validateFormOrRedirect($form)) {
            return $redirect;
        }
        $formJsons = json_decode($form->json);
        return view('form.integration.clockwork', compact('form', 'formJsons'));
    }

    public function salesforceIntegration($id)
    {
        $form = Form::find($id);
        if ($redirect = $this->validateFormOrRedirect($form)) {
            return $redirect;
        }
        $formJsons = json_decode($form->json);
        return view('form.integration.salesforce', compact('form', 'formJsons'));
    }

    public function hubspotIntegration($id)
    {
        $form = Form::find($id);
        if ($redirect = $this->validateFormOrRedirect($form)) {
            return $redirect;
        }
        $formJsons = json_decode($form->json);
        return view('form.integration.hubspot', compact('form', 'formJsons'));
    }

    public function pipedriveIntegration($id)
    {
        $form = Form::find($id);
        if ($redirect = $this->validateFormOrRedirect($form)) {
            return $redirect;
        }
        $formJsons = json_decode($form->json);
        return view('form.integration.pipedrive', compact('form', 'formJsons'));
    }

    public function formpaymentIntegration(Request $request, $id)
    {
        $form = Form::find($id);
        if ($redirect = $this->validateFormOrRedirect($form)) {
            return $redirect;
        }
        $paymentType = [];
        $paymentType[''] = 'Select payment';
        if (UtilityFacades::getsettings('stripesetting') == 'on') {
            $paymentType['stripe'] = 'Stripe';
        }
        if (UtilityFacades::getsettings('paypalsetting') == 'on') {
            $paymentType['paypal'] = 'Paypal';
        }
        if (UtilityFacades::getsettings('razorpaysetting') == 'on') {
            $paymentType['razorpay'] = 'Razorpay';
        }
        if (UtilityFacades::getsettings('paytmsetting') == 'on') {
            $paymentType['paytm'] = 'Paytm';
        }
        if (UtilityFacades::getsettings('flutterwavesetting') == 'on') {
            $paymentType['flutterwave'] = 'Flutterwave';
        }
        if (UtilityFacades::getsettings('paystacksetting') == 'on') {
            $paymentType['paystack'] = 'Paystack';
        }
        if (UtilityFacades::getsettings('payumoneysetting') == 'on') {
            $paymentType['payumoney'] = 'PayuMoney';
        }
        if (UtilityFacades::getsettings('molliesetting') == 'on') {
            $paymentType['mollie'] = 'Mollie';
        }
        if (UtilityFacades::getsettings('coingatesetting') == 'on') {
            $paymentType['coingate'] = 'Coingate';
        }
        if (UtilityFacades::getsettings('mercadosetting') == 'on') {
            $paymentType['mercado'] = 'Mercado';
        }
        if (UtilityFacades::getsettings('offlinepaymentsetting') == 'on') {
            $paymentType['offlinepayment'] = 'offlinepayment';
        }
        return view('form.payment', compact('form', 'paymentType'));
    }

    public function formpaymentIntegrationstore(Request $request, $id)
    {
        $form = Form::find($id);
        if ($redirect = $this->validateFormOrRedirect($form)) {
            return $redirect;
        }
        if ($request->payment_type == "paystack") {
            if ($request->currency_symbol != '₦' || $request->currency_name != 'NGN') {
                return redirect()->back()->with('failed', __('Currency not suppoted this payment getway. please enter NGN currency and ₦ symbol.'));
            }
        }
        if ($request->payment_type == "paytm") {
            if ($request->currency_symbol != '₹' || $request->currency_name != 'INR') {
                return redirect()->back()->with('failed', __('Currency not suppoted this payment getway. please enter INR currency and ₹ symbol.'));
            }
        }
        if ($request->payment_type == "coingate") {
            $allowedCoinGateCurrencies = [
                'USD', 'EUR', 'GBP', 'BTC', 'USDT'
            ];

            if (!in_array($request->currency_name, $allowedCoinGateCurrencies)) {
                return redirect()->back()->with('failed', __('Currency not supported by CoinGate. Please select a valid currency.'));
            }
        }
        $form->payment_status   = ($request->payment == 'on') ? '1' : '0';
        $form->amount           = ($request->amount == '') ? '0' : $request->amount;
        $form->currency_symbol  = $request->currency_symbol;
        $form->currency_name    = $request->currency_name;
        $form->payment_type     = $request->payment_type;
        $form->save();
        return redirect()->back()->with('success', __('Form payment integration succesfully.'));
    }

    public function formIntegrationStore(Request $request, $id)
    {
        $slackdata = [];
        $slackFiledtext = [];
        if ($request->get('slack_webhook_url')) {
            foreach ($request->get('slack_webhook_url') as $slackkey => $slackvalue) {
                $slackdata[$slackkey]['slack_webhook_url'] = $slackvalue;
                $slackField                                = $request->input('slack_field' . $slackkey);
                if ($slackField) {
                    $slackFiledtext[] = implode(',', $slackField);
                }
            }
        }
        $slackJsonData = ($slackdata) ? json_encode($slackdata) : null;
        FormIntegrationSetting::updateOrCreate(
            ['form_id' => $id,  'key' => 'slack_integration'],
            ['status' => ($request->get('slack_webhook_url')) ? 1 : 0, 'field_json' => json_encode($slackFiledtext), 'json' => $slackJsonData]
        );
        $telegramdata = [];
        $telegramFiledtext = [];
        if ($request->get('telegram_access_token') && $request->get('telegram_chat_id')) {
            foreach ($request->get('telegram_access_token') as $telegramkey => $telegramvalue) {
                $telegramdata[$telegramkey]['telegram_access_token'] = $telegramvalue;
                $telegramdata[$telegramkey]['telegram_chat_id']      = $request->get('telegram_chat_id')[$telegramkey];
                $telegramField                                       = $request->input('telegram_field' . $telegramkey);
                if ($telegramField) {
                    $telegramFiledtext[] = implode(',', $telegramField);
                }
            }
        }
        $telegramJsonData = ($telegramdata) ? json_encode($telegramdata) : null;
        FormIntegrationSetting::updateorcreate(
            ['form_id' => $id,  'key' => 'telegram_integration'],
            ['status' => ($request->get('telegram_access_token') && $request->get('telegram_chat_id')) ? 1 : 0, 'field_json' => json_encode($telegramFiledtext), 'json' => $telegramJsonData]
        );

        $mailgundata = [];
        $mailgunFiledtext = [];
        if ($request->get('mailgun_email') && $request->get('mailgun_domain') && $request->get('mailgun_secret')) {
            foreach ($request->get('mailgun_email') as $mailgunkey => $mailgunvalue) {
                $mailgundata[$mailgunkey]['mailgun_email']  = $mailgunvalue;
                $mailgundata[$mailgunkey]['mailgun_domain'] = $request->get('mailgun_domain')[$mailgunkey];
                $mailgundata[$mailgunkey]['mailgun_secret'] = $request->get('mailgun_secret')[$mailgunkey];
                $mailgunField                               = $request->input('mailgun_field' . $mailgunkey);
                if ($mailgunField) {
                    $mailgunFiledtext[] = implode(',', $mailgunField);
                }
            }
        }
        $mailgunJsonData = ($mailgundata) ? json_encode($mailgundata) : null;
        FormIntegrationSetting::updateorcreate(
            ['form_id' => $id,  'key' => 'mailgun_integration'],
            ['status' => ($request->get('mailgun_email') && $request->get('mailgun_domain') && $request->get('mailgun_secret')) ? 1 : 0, 'field_json' => json_encode($mailgunFiledtext), 'json' => $mailgunJsonData]
        );

        $bulkgatedata = [];
        $bulkgateFiledtext = [];
        if ($request->get('bulkgate_number') && $request->get('bulkgate_token') && $request->get('bulkgate_app_id')) {
            foreach ($request->get('bulkgate_number') as $bulkgatekey => $bulkgatevalue) {
                $bulkgatedata[$bulkgatekey]['bulkgate_number'] = $bulkgatevalue;
                $bulkgatedata[$bulkgatekey]['bulkgate_token']  = $request->get('bulkgate_token')[$bulkgatekey];
                $bulkgatedata[$bulkgatekey]['bulkgate_app_id'] = $request->get('bulkgate_app_id')[$bulkgatekey];
                $bulkgateField                                 = $request->input('bulkgate_field' . $bulkgatekey);
                if ($bulkgateField) {
                    $bulkgateFiledtext[] = implode(',', $bulkgateField);
                }
            }
        }
        $bulkgateJsonData = ($bulkgatedata) ? json_encode($bulkgatedata) : null;
        FormIntegrationSetting::updateorcreate(
            ['form_id' => $id,  'key' => 'bulkgate_integration'],
            ['status' => ($request->get('bulkgate_number') && $request->get('bulkgate_token') && $request->get('bulkgate_app_id')) ? 1 : 0, 'field_json' => json_encode($bulkgateFiledtext), 'json' => $bulkgateJsonData]
        );

        $nexmodata = [];
        $nexmoFiledtext = [];
        if ($request->get('nexmo_number') && $request->get('nexmo_key') && $request->get('nexmo_secret')) {
            foreach ($request->get('nexmo_number') as $nexmokey => $nexmovalue) {
                $nexmodata[$nexmokey]['nexmo_number'] = $nexmovalue;
                $nexmodata[$nexmokey]['nexmo_key']    = $request->get('nexmo_key')[$nexmokey];
                $nexmodata[$nexmokey]['nexmo_secret'] = $request->get('nexmo_secret')[$nexmokey];
                $nexmoField                           = $request->input('nexmo_field' . $nexmokey);
                if ($nexmoField) {
                    $nexmoFiledtext[] = implode(',', $nexmoField);
                }
            }
        }
        $nexmoJsonData = ($nexmodata) ? json_encode($nexmodata) : null;
        FormIntegrationSetting::updateorcreate(
            ['form_id' => $id,  'key' => 'nexmo_integration'],
            ['status' => ($request->get('nexmo_number') && $request->get('nexmo_key') && $request->get('nexmo_secret')) ? 1 : 0, 'field_json' => json_encode($nexmoFiledtext), 'json' => $nexmoJsonData]
        );
        $fast2smsdata = [];
        $fast2smsFiledtext = [];
        if ($request->get('fast2sms_number') && $request->get('fast2sms_api_key')) {
            foreach ($request->get('fast2sms_number') as $fast2smskey => $fast2smsvalue) {
                $fast2smsdata[$fast2smskey]['fast2sms_number']  = $fast2smsvalue;
                $fast2smsdata[$fast2smskey]['fast2sms_api_key'] = $request->input('fast2sms_api_key')[$fast2smskey];
                $fast2smsField                                  = $request->input('fast2sms_field' . $fast2smskey);
                if ($fast2smsField) {
                    $fast2smsFiledtext[] = implode(',', $fast2smsField);
                }
            }
        }
        $fast2smsJsonData = ($fast2smsdata) ? json_encode($fast2smsdata) : null;
        FormIntegrationSetting::updateorcreate(
            ['form_id' => $id,  'key' => 'fast2sms_integration'],
            ['status' => ($request->get('fast2sms_number') && $request->get('fast2sms_api_key')) ? 1 : 0, 'field_json' => json_encode($fast2smsFiledtext), 'json' => $fast2smsJsonData]
        );

        $vonagedata = [];
        $vonageFiledtext = [];
        if ($request->get('vonage_number') && $request->get('vonage_key') && $request->get('vonage_secret')) {
            foreach ($request->get('vonage_number') as $vonagekey => $vonagevalue) {
                $vonagedata[$vonagekey]['vonage_number'] = $vonagevalue;
                $vonagedata[$vonagekey]['vonage_key']    = $request->input('vonage_key')[$vonagekey];
                $vonagedata[$vonagekey]['vonage_secret'] = $request->input('vonage_secret')[$vonagekey];
                $vonageField                             = $request->input('vonage_field' . $vonagekey);
                if ($vonageField) {
                    $vonageFiledtext[] = implode(',', $vonageField);
                }
            }
        }
        $vonageJsonData = ($vonagedata) ? json_encode($vonagedata) : null;
        FormIntegrationSetting::updateorcreate(
            ['form_id' => $id,  'key' => 'vonage_integration'],
            ['status' => ($request->get('vonage_number') && $request->get('vonage_key') && $request->get('vonage_secret')) ? 1 : 0, 'field_json' => json_encode($vonageFiledtext), 'json' => $vonageJsonData]
        );

        $sendgriddata = [];
        $sendgridFiledtext = [];
        if ($request->get('sendgrid_email') && $request->get('sendgrid_host') && $request->get('sendgrid_port') && $request->get('sendgrid_username') && $request->get('sendgrid_password') && $request->get('sendgrid_encryption') && $request->get('sendgrid_from_address') && $request->get('sendgrid_from_name')) {
            foreach ($request->get('sendgrid_email') as $sendgridkey => $sendgridvalue) {
                $sendgriddata[$sendgridkey]['sendgrid_email']      = $sendgridvalue;
                $sendgriddata[$sendgridkey]['sendgrid_host']       = $request->get('sendgrid_host')[$sendgridkey];
                $sendgriddata[$sendgridkey]['sendgrid_port']       = $request->get('sendgrid_port')[$sendgridkey];
                $sendgriddata[$sendgridkey]['sendgrid_username']   = $request->get('sendgrid_username')[$sendgridkey];
                $sendgriddata[$sendgridkey]['sendgrid_password']   = $request->get('sendgrid_password')[$sendgridkey];
                $sendgriddata[$sendgridkey]['sendgrid_encryption'] = $request->get('sendgrid_encryption')[$sendgridkey];
                $sendgriddata[$sendgridkey]['sendgrid_from_address'] = $request->get('sendgrid_from_address')[$sendgridkey];
                $sendgriddata[$sendgridkey]['sendgrid_from_name'] = $request->get('sendgrid_from_name')[$sendgridkey];
                $sendgridField                                    = $request->input('sendgrid_field' . $sendgridkey);
                if ($sendgridField) {
                    $sendgridFiledtext[] = implode(',', $sendgridField);
                }
            }
        }
        $sendgridJsonData = ($sendgriddata) ? json_encode($sendgriddata) : null;
        FormIntegrationSetting::updateorcreate(
            ['form_id' => $id,  'key' => 'sendgrid_integration'],
            ['status' => ($request->get('sendgrid_email') && $request->get('sendgrid_host') && $request->get('sendgrid_port') && $request->get('sendgrid_username') && $request->get('sendgrid_password') && $request->get('sendgrid_encryption') && $request->get('sendgrid_from_address') && $request->get('sendgrid_from_name')) ? 1 : 0, 'field_json' => json_encode($sendgridFiledtext), 'json' => $sendgridJsonData]
        );

        $twiliodata = [];
        $twilioFiledtext = [];
        if ($request->get('twilio_mobile_number') && $request->get('twilio_sid') && $request->get('twilio_auth_token') && $request->get('twilio_number')) {
            foreach ($request->get('twilio_mobile_number') as $twiliokey => $twiliovalue) {
                $twiliodata[$twiliokey]['twilio_mobile_number'] = $twiliovalue;
                $twiliodata[$twiliokey]['twilio_sid']        = $request->get('twilio_sid')[$twiliokey];
                $twiliodata[$twiliokey]['twilio_auth_token'] = $request->get('twilio_auth_token')[$twiliokey];
                $twiliodata[$twiliokey]['twilio_number']     = $request->get('twilio_number')[$twiliokey];
                $twilioField                                 = $request->input('twilio_field' . $twiliokey);
                if ($twilioField) {
                    $twilioFiledtext[] = implode(',', $twilioField);
                }
            }
        }
        $twilioJsonData = ($twiliodata) ? json_encode($twiliodata) : null;
        FormIntegrationSetting::updateorcreate(
            ['form_id' => $id,  'key' => 'twilio_integration'],
            ['status' => ($request->get('twilio_mobile_number') && $request->get('twilio_sid') && $request->get('twilio_auth_token') && $request->get('twilio_number')) ? 1 : 0, 'field_json' => json_encode($twilioFiledtext), 'json' => $twilioJsonData]
        );

        $textlocaldata = [];
        $textlocalFiledtext = [];
        if ($request->get('textlocal_number') && $request->get('textlocal_api_key')) {
            foreach ($request->get('textlocal_number') as $textlocalkey => $textlocalvalue) {
                $textlocaldata[$textlocalkey]['textlocal_number']  = $textlocalvalue;
                $textlocaldata[$textlocalkey]['textlocal_api_key'] = $request->input('textlocal_api_key')[$textlocalkey];
                $textlocalField                                    = $request->input('textlocal_field' . $textlocalkey);
                if ($textlocalField) {
                    $textlocalFiledtext[] = implode(',', $textlocalField);
                }
            }
        }
        $textlocalJsonData = ($textlocaldata) ? json_encode($textlocaldata) : null;
        FormIntegrationSetting::updateorcreate(
            ['form_id' => $id,  'key' => 'textlocal_integration'],
            ['status' => ($request->get('textlocal_number') && $request->get('textlocal_api_key')) ? 1 : 0, 'field_json' => json_encode($textlocalFiledtext), 'json' => $textlocalJsonData]
        );

        $messentedata = [];
        $messenteFiledtext = [];
        if ($request->get('messente_number') && $request->get('messente_api_username') && $request->get('messente_api_password') && $request->get('messente_sender')) {
            foreach ($request->get('messente_number') as $messentekey => $messentevalue) {
                $messentedata[$messentekey]['messente_number']       = $messentevalue;
                $messentedata[$messentekey]['messente_api_username'] = $request->get('messente_api_username')[$messentekey];
                $messentedata[$messentekey]['messente_api_password'] = $request->get('messente_api_password')[$messentekey];
                $messentedata[$messentekey]['messente_sender'] = $request->get('messente_sender')[$messentekey];
                $messenteField                                 = $request->input('messente_field' . $messentekey);
                if ($messenteField) {
                    $messenteFiledtext[] = implode(',', $messenteField);
                }
            }
        }
        $messenteJsonData = ($messentedata) ? json_encode($messentedata) : null;
        FormIntegrationSetting::updateorcreate(
            ['form_id' => $id,  'key' => 'messente_integration'],
            ['status' => ($request->get('messente_number') && $request->get('messente_api_username') && $request->get('messente_api_password') && $request->get('messente_sender')) ? 1 : 0, 'field_json' => json_encode($messenteFiledtext), 'json' => $messenteJsonData]
        );

        $smsgatewaydata = [];
        $smsgatewayFiledtext = [];
        if ($request->get('smsgateway_number') && $request->get('smsgateway_api_key') && $request->get('smsgateway_user_id') && $request->get('smsgateway_user_password') && $request->get('smsgateway_sender_id')) {
            foreach ($request->get('smsgateway_number') as $smsgatewaykey => $smsgatewayvalue) {
                $smsgatewaydata[$smsgatewaykey]['smsgateway_number']    = $smsgatewayvalue;
                $smsgatewaydata[$smsgatewaykey]['smsgateway_api_key']   = $request->get('smsgateway_api_key')[$smsgatewaykey];
                $smsgatewaydata[$smsgatewaykey]['smsgateway_user_id']   = $request->get('smsgateway_user_id')[$smsgatewaykey];
                $smsgatewaydata[$smsgatewaykey]['smsgateway_user_password']       = $request->get('smsgateway_user_password')[$smsgatewaykey];
                $smsgatewaydata[$smsgatewaykey]['smsgateway_sender_id'] = $request->get('smsgateway_sender_id')[$smsgatewaykey];
                $smsgatewayField                                        = $request->input('smsgateway_field' . $smsgatewaykey);
                if ($smsgatewayField) {
                    $smsgatewayFiledtext[] = implode(',', $smsgatewayField);
                }
            }
        }
        $smsgatewayJsonData = ($smsgatewaydata) ? json_encode($smsgatewaydata) : null;
        FormIntegrationSetting::updateorcreate(
            ['form_id' => $id,  'key' => 'smsgateway_integration'],
            ['status' => ($request->get('smsgateway_number') && $request->get('smsgateway_user_id') && $request->get('smsgateway_user_password') && $request->get('smsgateway_sender_id')) ? 1 : 0, 'field_json' => json_encode($smsgatewayFiledtext), 'json' => $smsgatewayJsonData]
        );
        $clicktelldata = [];
        $clicktellFiledtext = [];
        if ($request->get('clicktell_number') && $request->get('clicktell_api_key')) {
            foreach ($request->get('clicktell_number') as $clicktellkey => $clicktellvalue) {
                $clicktelldata[$clicktellkey]['clicktell_number']  = $clicktellvalue;
                $clicktelldata[$clicktellkey]['clicktell_api_key'] = $request->get('clicktell_api_key')[$clicktellkey];
                $clicktellField                                    = $request->input('clicktell_field' . $clicktellkey);
                if ($clicktellField) {
                    $clicktellFiledtext[] = implode(',', $clicktellField);
                }
            }
        }
        $clicktellJsonData = ($clicktelldata) ? json_encode($clicktelldata) : null;
        FormIntegrationSetting::updateorcreate(
            ['form_id' => $id,  'key' => 'clicktell_integration'],
            ['status' => ($request->get('clicktell_number') && $request->get('clicktell_api_key')) ? 1 : 0, 'field_json' => json_encode($clicktellFiledtext), 'json' => $clicktellJsonData]
        );

        $clockworkdata = [];
        $clockworkFiledtext = [];
        if ($request->get('clockwork_number') && $request->get('clockwork_api_token')) {
            foreach ($request->get('clockwork_number') as $clockworkkey => $clockworkvalue) {
                $clockworkdata[$clockworkkey]['clockwork_number']    = $clockworkvalue;
                $clockworkdata[$clockworkkey]['clockwork_api_token'] = $request->input('clockwork_api_token')[$clockworkkey];
                $clockworkField                                      = $request->input('clockwork_field' . $clockworkkey);
                if ($clockworkField) {
                    $clockworkFiledtext[] = implode(',', $clockworkField);
                }
            }
        }
        $clockworkJsonData = ($clockworkdata) ? json_encode($clockworkdata) : null;
        FormIntegrationSetting::updateorcreate(
            ['form_id' => $id,  'key' => 'clockwork_integration'],
            ['status' => ($request->get('clockwork_number') && $request->get('clockwork_api_token')) ? 1 : 0, 'field_json' => json_encode($clockworkFiledtext), 'json' => $clockworkJsonData]
        );

        $hubspotdata = [];
        $hubspotFiledtext = [];
        if ($request->get('hubspot_access_token')) {
            foreach ($request->get('hubspot_access_token') as $hubspotkey => $hubspotvalue) {
                $hubspotdata[$hubspotkey]['hubspot_access_token'] = $request->input('hubspot_access_token')[$hubspotkey];
                $hubspotField                                     = $request->input('hubspot_field' . $hubspotkey);
                if ($hubspotField) {
                    $hubspotFiledtext[] = implode(',', $hubspotField);
                }
            }
        }
        $hubspotJsonData = ($hubspotdata) ? json_encode($hubspotdata) : null;
        FormIntegrationSetting::updateorcreate(
            ['form_id' => $id,  'key' => 'hubspot_integration'],
            ['status' => ($request->get('hubspot_access_token')) ? 1 : 0, 'field_json' => json_encode($hubspotFiledtext), 'json' => $hubspotJsonData]
        );

        $pipedrivedata = [];
        $pipedriveFiledtext = [];
        if ($request->get('pipedrive_api_token')) {
            foreach ($request->get('pipedrive_api_token') as $pipedrivekey => $pipedrivevalue) {
                $pipedrivedata[$pipedrivekey]['pipedrive_api_token']    = $request->input('pipedrive_api_token')[$pipedrivekey];
                $typeArray = [];
                foreach ($request->input('pipedrive_type') as $type) {
                    $typeArray[] = $type;
                }
                $pipedrivedata[$pipedrivekey]['pipedrive_type']   = $typeArray;

                $pipedriveField = $request->input('pipedrive_field' . $pipedrivekey);
                if ($pipedriveField) {
                    $pipedriveFiledtext[] = implode(',', $pipedriveField);
                }
            }
        }
        $pipedriveJsonData = ($pipedrivedata) ? json_encode($pipedrivedata) : null;
        FormIntegrationSetting::updateorcreate(
            ['form_id' => $id,  'key' => 'pipedrive_integration'],
            ['status' => ($request->get('pipedrive_api_token')) ? 1 : 0, 'field_json' => json_encode($pipedriveFiledtext), 'json' => $pipedriveJsonData]
        );

        $salesforcedata = [];
        $salesforceFiledtext = [];
        if ($request->get('sf_consumer_key') && $request->get('sf_consumer_secret')) {
            foreach ($request->get('sf_consumer_key') as $salesforcekey => $salesforcevalue) {
                $salesforcedata[$salesforcekey]['sf_consumer_key']    =  $request->input('sf_consumer_key')[$salesforcekey];
                $salesforcedata[$salesforcekey]['sf_consumer_secret'] =  $request->input('sf_consumer_secret')[$salesforcekey];
                $salesforcedata[$salesforcekey]['sf_callback_uri']    =  $request->input('sf_callback_uri')[$salesforcekey];
                $salesforceField                                      = $request->input('salesforce_field' . $salesforcekey);
                if ($salesforceField) {
                    $salesforceFiledtext[] = implode(',', $salesforceField);
                }
            }
        }
        $salesforceJsonData = ($salesforcedata) ? json_encode($salesforcedata) : null;
        FormIntegrationSetting::updateorcreate(
            ['form_id' => $id,  'key' => 'salesforce_integration'],
            ['status' => ($request->get('sf_consumer_key') && $request->get('sf_consumer_secret')) ? 1 : 0, 'field_json' => json_encode($salesforceFiledtext), 'json' => $salesforceJsonData]
        );


        return redirect()->back()->with('success', __('Form integration succesfully.'));
    }

    public function formTheme($id)
    {
        $form = Form::find($id);
        if ($redirect = $this->validateFormOrRedirect($form)) {
            return $redirect;
        }
        return view('form.themes.theme', compact('form'));
    }

    public function formThemeEdit(Request $request, $slug, $id)
    {
        $form = Form::find($id);
        if ($redirect = $this->validateFormOrRedirect($form)) {
            return $redirect;
        }
        return view('form.themes.index', compact('slug', 'form'));
    }

    public function themeChange(Request $request, $id)
    {
        $form = Form::find($id);
        if ($redirect = $this->validateFormOrRedirect($form)) {
            return $redirect;
        }
        $form->theme = $request->theme;
        $form->save();
        return redirect()->route('forms.index')->with('success', __('Theme successfully changed.'));
    }

    public function formThemeUpdate(Request $request, $id)
    {
        $validator = \Validator::make($request->all(), [
            'background_image' => 'image|mimes:png,jpg,jpeg',
        ]);
        if ($validator->fails()) {
            $messages = $validator->errors();
            return response()->json(['errors' => $messages->first()]);
        }
        if (isset($request->color) && $request->color_flag == 'false') {
            $color = $request->color;
        } else {
            $color = $request->custom_color;
        }
        $form = Form::find($id);
        if ($redirect = $this->validateFormOrRedirect($form)) {
            return $redirect;
        }
        $form->theme       = $request->theme;
        $form->theme_color = $color;
        $form->color_flag  = $request->color_flag;
        if ($request->hasFile('background_image')) {
            $themeBackgroundImage     = 'form-background.' . $request->background_image->getClientOriginalExtension();
            $themeBackgroundImagePath = 'form-themes/theme3/' . $form->id;
            $backgroundImage          = $request->file('background_image')->storeAs(
                $themeBackgroundImagePath,
                $themeBackgroundImage
            );
            $form->theme_background_image = $backgroundImage;
        }
        $form->save();
        return redirect()->route('forms.index')->with('success', __('Form theme selected succesfully.'));
    }

    public function formRules(Request $request,  $id)
    {
        if (Auth::user()->can('manage-form-rule')) {
            $formRules = form::find($id);
            if ($redirect = $this->validateFormOrRedirect($formRules)) {
                return $redirect;
            }
            $jsonData = json_decode($formRules->json);
            $rules    = formRule::where('form_id', $id)->get();
            return view('form.conditional-rules.rules', compact('formRules', 'jsonData', 'rules'));
        } else {
            return redirect()->back()->with('failed', __('permission Denied'));
        }
    }

    public function storeRule(Request $request)
    {
        if (Auth::user()->can('create-form-rule')) {

            request()->validate([
                'rule_name'      => 'required|max:50',
                'condition_type' => 'nullable',
                'rules.*.if_field_name' => 'required',
                'rules.*.if_rule_type'  => 'required',
                'rules.*.if_rule_value' => 'required',
                'rules2.*.else_rule_type'  => 'required',
                'rules2.*.else_field_name' => 'required',
            ]);

            $conditioal = Form::find($request->form_id);
            $conditioal->conditional_rule = ($request->conditional_rule  == '1' ? '1'  : '0');
            $conditioal->save();

            $ifJson   = json_encode($request->rules);
            $thenJson = json_encode($request->rules2);

            $formRule            = new formRule();
            $formRule->form_id   = $request->form_id;
            $formRule->rule_name = $request->rule_name;
            $formRule->if_json   = $ifJson;
            $formRule->then_json = $thenJson;
            $formRule->condition = ($request->condition_type) ?  $request->condition_type : 'or';
            $formRule->save();

            return redirect()->route('form.rules', $request->form_id)->with('success', __('Rule set successfully'));
        } else {
            return redirect()->back()->with('failed', __('permission Denied'));
        }
    }

    public function editRule($id)
    {
        if (Auth::user()->can('edit-form-rule')) {
            $rule       = formRule::where('id', $id)->first();
            $form       = form::find($rule->form_id);
            $jsonDataIf = json_decode($rule->if_json);
            $jsonDataThen = json_decode($rule->then_json);
            $jsonData     = json_decode($form->json);

            return view('form.conditional-rules.edit', compact('form', 'rule', 'jsonDataIf', 'jsonDataThen', 'jsonData'));
        } else {
            return redirect()->back()->with('failed', __('permission Denied'));
        }
    }

    public function ruleUpdate($id, Request $request)
    {
        if (Auth::user()->can('edit-form-rule')) {
            request()->validate([
                'rule_name'      => 'required|max:50',
                'condition_type' => 'nullable',
                'rules.*.if_field_name' => 'required',
                'rules.*.if_rule_type'  => 'required',
                'rules.*.if_rule_value' => 'required',
                'rules2.*.else_rule_type'   => 'required',
                'rules2.*.else_field_name'  => 'required',
            ]);

            $conditioal = Form::find($request->form_id);
            $conditioal->conditional_rule = ($request->conditional_rule     == 'on' ? '1'  : '0');
            $conditioal->save();

            $newRules       = $request->rules;
            $existingRules  = formRule::find($id)->if_json;
            $existingRules  = json_decode($existingRules, true);

            $countNewRules = count($newRules);
            $countExistingRules = count($existingRules);

            $lastPosition   = count($newRules) - 1;
            $lastRule       = $newRules[$lastPosition];

            if ($countExistingRules < $countNewRules) {
                foreach ($newRules as $newRule) {
                    $newFieldName = $lastRule['if_field_name'];
                    foreach ($existingRules as $existingRule) {
                        $existingFieldName = $existingRule['if_field_name'];

                        if ($newFieldName === $existingFieldName) {
                            return redirect()->back()->with('failed', 'This name Rule already exists.');
                        }
                    }
                }
            }

            $ifJson   = json_encode($request->rules);
            $thenJson = json_encode($request->rules2);

            $ruleUpdate = formRule::find($id);
            $ruleUpdate->rule_name  = $request->rule_name;
            $ruleUpdate->if_json    = $ifJson;
            $ruleUpdate->then_json  = $thenJson;
            $ruleUpdate->condition  = ($request->condition_type) ?  $request->condition_type : 'or';
            $ruleUpdate->save();

            return redirect()->route('form.rules', $request->form_id)->with('success', __('Rule set successfully'));
        } else {
            return redirect()->back()->with('failed', __('permission Denied'));
        }
    }

    public function ruleDelete($id)
    {
        if (Auth::user()->can('delete-form-rule')) {
            $ruleDelete  = formRule::find($id);
            $ruleDelete->delete();

            return back()->with('success', __('Rule Deleted Succesfully'));
        } else {
            return redirect()->back()->with('failed', __('permission Denied'));
        }
    }

    public function getField(Request $request)
    {
        $form = Form::find($request->id);
        $formData = json_decode($form->json, true);
        $fieldName = $request->input('fieldname');

        $matchingField = null;
        foreach ($formData as $section) {
            foreach ($section as $field) {
                if (isset($field['name']) && $field['name'] === $fieldName) {
                    $matchingField = $field;
                    break 2;
                }
            }
        }

        return response(['matchingField' => $matchingField]);
    }

    public function passwordProtection($id)
    {
        $hashids = new Hashids('', 20);
        $DechashId = $hashids->decodeHex($id);
        $form = Form::find($DechashId);
        return view('form.password-protection', compact('form'));
    }

    public function passwordMatchProtecrtion(Request $request,  $id)
    {
        request()->validate([
            'form_password' => 'required',
        ]);

        $form = Form::find($id);
        if ($redirect = $this->validateFormOrRedirect($form)) {
            return $redirect;
        }
        $hashids = new Hashids('', 20);
        $hashId = $hashids->encodeHex($form->id);

        if (Hash::check($request->form_password, $form->form_password)) {
            $request->session()->put('form_unlocked_' . $hashId, true);
            return redirect()->route('forms.survey', $hashId);
        } else {
            return redirect()->back()->with('failed', 'Password Doesn’t Match.! ');
        }
    }

    public function destroyMultiple(Request $request)
    {
        if (Auth::user()->can('delete-form')) {
            Feedback::whereIn('form_id', $request->ids)->delete();
            Form::whereIn('id', $request->ids)->delete();
            FormValue::whereIn('form_id', $request->ids)->delete();
            return response()->json(['msg' =>  'Form moved to trash']);
        } else {
            return redirect()->back()->with('failed', __('Permission Denied.'));
        }
    }

    public function restore($id)
    {
        if ($form = Form::onlyTrashed()->find($id)) {
            Feedback::onlyTrashed()->where('form_id', $form->id)->restore();
            FormValue::onlyTrashed()->where('form_id', $form->id)->restore();
            $form->restore();
            $message = ['success' => __('Form restored successfully')];
        } else {
            $message = ['failed' => __('Form not found.')];
        }

        return redirect()->route('forms.index', 'view=trash')->with($message);
    }

    public function restoreMultiple(Request $request)
    {
        $ids = $request->input('ids', []);
        if (!empty($ids)) {
            Form::onlyTrashed()->whereIn('id', $request->ids)->restore();
            FormValue::onlyTrashed()->whereIn('form_id', $request->ids)->restore();
            Feedback::onlyTrashed()->whereIn('form_id', $request->ids)->restore();
        }
        return response()->json(['msg' =>  'form restored successfully']);
    }

    public function forcedelete($id)
    {
        $form = Form::onlyTrashed()->find($id);
        if ($form) {
            AssignFormsRoles::where('form_id', $form->id)->delete();
            AssignFormsUsers::where('form_id', $form->id)->delete();
            DashboardWidget::where('form_id', $form->id)->delete();
            Feedback::onlyTrashed()->where('form_id', $form->id)->forceDelete();
            FormComments::where('form_id', $form->id)->delete();
            FormCommentsReply::where('form_id', $form->id)->delete();
            FormValue::onlyTrashed()->where('form_id', $form->id)->forceDelete();
            $form->forceDelete();
            $message = ['success' => __('Form deleted successfully')];
        } else {
            $message = ['failed' => __('Form not found.')];
        }

        return redirect()->route('forms.index', 'view=trash')->with($message);
    }

    public function forcedeleteMultiple(Request $request)
    {
        $request->validate([
            'ids' => 'required|array',
        ]);
        $ids = $request->ids;
        Form::whereIn('id', $ids)->forceDelete();
        AssignFormsRoles::whereIn('form_id', $ids)->delete();
        AssignFormsUsers::whereIn('form_id', $ids)->delete();
        DashboardWidget::whereIn('form_id', $ids)->delete();
        Feedback::whereIn('form_id', $ids)->forceDelete();
        FormComments::whereIn('form_id', $ids)->delete();
        FormCommentsReply::whereIn('form_id', $ids)->delete();
        FormValue::whereIn('form_id', $ids)->forceDelete();

        return response()->json(['msg' => __('Forms deteted successfully.')]);
        if ($request->query->get('view')) {
            return route('forms.index', 'view=trash');
        } else {
            return route('forms.index');
        }
    }

    public function forcedeleteAll(Request $request)
    {
        $forms = Form::onlyTrashed()->get();
        foreach ($forms as $form) {
            if ($form) {
                AssignFormsRoles::where('form_id', $form->id)->delete();
                AssignFormsUsers::where('form_id', $form->id)->delete();
                DashboardWidget::where('form_id', $form->id)->delete();
                Feedback::onlyTrashed()->where('form_id', $form->id)->forceDelete();
                FormComments::where('form_id', $form->id)->delete();
                FormCommentsReply::where('form_id', $form->id)->delete();
                FormValue::onlyTrashed()->where('form_id', $form->id)->forceDelete();
                $form->forceDelete();
            }
        }
        Form::onlyTrashed()->forceDelete();
        return response()->json(['msg' => __('Trash is empty now.')]);
    }

    public function removeUpoadedFile($img, $formId)
    {
        $img = Crypt::decryptString($img);
        $formValue     = FormValue::find($formId);
        if (!$formValue) {
            return response()->json(['is_success' => false, 'message' => 'Submitted Form not found'], 404);
        }
        $formValueJson = json_decode($formValue->json);
        foreach ($formValueJson as $rows) {
            foreach ($rows as $row) {
                if (isset($row->type) && $row->type == 'file' && $row->multiple && isset($row->value)) {
                    foreach ($row->value as $key => $imaValue) {
                        if (isset($imaValue) && $imaValue === $img) {
                            unset($row->value[$key]);
                            if (Storage::exists($imaValue)) {
                                Storage::delete($imaValue);
                            }
                            $row->value = array_values($row->value);
                            $formValue->json = json_encode($formValueJson);
                            $formValue->save();
                            return response()->json(['is_success' => true, 'message' => 'Attachment deleted successfully'], 200);
                        }
                    }
                }
            }
        }
        return response()->json(['is_success' => false, 'message' => 'File not found in submitted form'], 404);
    }

    public static function validateFormOrRedirect($form)
    {
        if (!$form || empty($form)) {
            return redirect()->back()->with('failed', __('Form not found.'));
        }

        return null;
    }
}
